from fractal_lsystem_designer import LSystemRule, LSystemInterpreter
import matplotlib.pyplot as plt
from matplotlib.collections import LineCollection
import math
import numpy as np
import os

# Create output directory if it doesn't exist
os.makedirs('output', exist_ok=True)

class ColoredFractalVisualizer:
    
    @staticmethod
    def plot_lsystem_colored(lstring, angle_delta=90.0, step_length=1.0, 
                             title="", output_file=None, initial_angle=90.0,
                             colormap='viridis', linewidth=1.5):
        x, y = 0, 0
        angle = initial_angle * math.pi / 180
        stack = []
        points = [(x, y)]
        
        for char in lstring:
            if char in ['F', 'A', 'B', 'G', 'T']:
                x += step_length * math.cos(angle)
                y += step_length * math.sin(angle)
                points.append((x, y))
            elif char == '+':
                angle += angle_delta * math.pi / 180
            elif char == '-':
                angle -= angle_delta * math.pi / 180
            elif char == '[':
                stack.append((x, y, angle))
            elif char == ']':
                if stack:
                    x, y, angle = stack.pop()
                    points.append((x, y))
        
        points = np.array(points)
        
        segments = []
        for i in range(len(points) - 1):
            if np.linalg.norm(points[i+1] - points[i]) > 0:
                segments.append([points[i], points[i+1]])
        
        fig, ax = plt.subplots(figsize=(10, 10), facecolor='white')
        
        lc = LineCollection(segments, cmap=colormap, linewidth=linewidth)
        lc.set_array(np.linspace(0, 1, len(segments)))
        ax.add_collection(lc)
        
        ax.set_aspect('equal')
        
        ax.axis('off')
        
        ax.autoscale()
        ax.margins(0.05)
        
        if output_file:
            plt.savefig(output_file, dpi=150, bbox_inches='tight', facecolor='white', pad_inches=0.1)
            print(f"Saved {output_file}")
        
        plt.close(fig)
        return fig

print("Generating Koch Snowflake...")
koch_axiom = "F--F--F"
koch_rules = [LSystemRule("F", "F+F--F+F")]
koch_interpreter = LSystemInterpreter(angle_delta=60.0)
koch_lstring = koch_interpreter.generate_lstring(
    axiom=koch_axiom,
    rules=koch_rules,
    iterations=4
)
koch_dim = math.log(4) / math.log(3)

ColoredFractalVisualizer.plot_lsystem_colored(
    koch_lstring,
    angle_delta=60.0,
    step_length=1.0,
    title=f"Koch Snowflake (D = {koch_dim:.4f})",
    output_file='output/koch_snowflake.png',
    initial_angle=0.0,
    colormap='viridis',
    linewidth=2.0
)

print("Generating Sierpinski Triangle...")
sierpinski_axiom = "A"
sierpinski_rules = [
    LSystemRule("A", "B-A-B"),
    LSystemRule("B", "A+B+A")
]
sierpinski_interpreter = LSystemInterpreter(angle_delta=60.0)
sierpinski_lstring = sierpinski_interpreter.generate_lstring(
    axiom=sierpinski_axiom,
    rules=sierpinski_rules,
    iterations=6
)
sierpinski_dim = math.log(3) / math.log(2)

ColoredFractalVisualizer.plot_lsystem_colored(
    sierpinski_lstring,
    angle_delta=60.0,
    step_length=1.0,
    title=f"Sierpinski Triangle (D = {sierpinski_dim:.4f})",
    output_file='output/sierpinski_triangle.png',
    initial_angle=0.0,
    colormap='plasma',
    linewidth=1.5
)

print("Generating Hilbert-like Fractal...")
hilbert_axiom = "F"
hilbert_rules = [
    LSystemRule('F', 'F+F-F-FF+F+F-F')
]
hilbert_iterations = 3
hilbert_interpreter = LSystemInterpreter(angle_delta=90.0)
hilbert_lstring = hilbert_interpreter.generate_lstring(
    axiom=hilbert_axiom,
    rules=hilbert_rules,
    iterations=hilbert_iterations
)

ColoredFractalVisualizer.plot_lsystem_colored(
    hilbert_lstring,
    angle_delta=90.0,
    step_length=1.0,
    title="Hilbert-like Curve (D≈1.5)",
    output_file='output/custom_hilbert_like.png',
    initial_angle=0.0,
    colormap='inferno',
    linewidth=2.0
)

print("Generating Cantor-Tree Hybrid...")
cantor_axiom = 'T'
cantor_rules = [
    LSystemRule('T', 'F[+T]F[-T]G'),
    LSystemRule('G', 'F[+F]F[-F]')
]
cantor_iterations = 6
cantor_interpreter = LSystemInterpreter(angle_delta=20.0)
cantor_lstring = cantor_interpreter.generate_lstring(
    axiom=cantor_axiom,
    rules=cantor_rules,
    iterations=cantor_iterations
)

ColoredFractalVisualizer.plot_lsystem_colored(
    cantor_lstring,
    angle_delta=20.0,
    step_length=0.9,
    title="Cantor-Tree Hybrid Fractal (D≈1.70)",
    output_file='output/custom_cantor_tree.png',
    initial_angle=90.0,
    colormap='YlGn',
    linewidth=1.5
)

print("Generating Levy C Curve...")
levy_axiom = "F"
levy_rules = [
    LSystemRule('F', '+F--F+')
]
levy_iterations = 12
levy_interpreter = LSystemInterpreter(angle_delta=45.0)
levy_lstring = levy_interpreter.generate_lstring(
    axiom=levy_axiom,
    rules=levy_rules,
    iterations=levy_iterations
)
levy_dim = 2.0

ColoredFractalVisualizer.plot_lsystem_colored(
    levy_lstring,
    angle_delta=45.0,
    step_length=1.0,
    title=f"Levy C Curve (D={levy_dim:.1f})",
    output_file='output/custom_levy_c_curve.png',
    initial_angle=0.0,
    colormap='cool',
    linewidth=1.5
)

print("Generating Hexagonal Gosper Curve...")
gosper_axiom = "A"
gosper_rules = [
    LSystemRule('A', 'A-B--B+A++AA+B-'),
    LSystemRule('B', '+A-BB--B-A++A+B')
]
gosper_iterations = 4
gosper_interpreter = LSystemInterpreter(angle_delta=60.0)
gosper_lstring = gosper_interpreter.generate_lstring(
    axiom=gosper_axiom,
    rules=gosper_rules,
    iterations=gosper_iterations
)
gosper_dim = math.log(7) / math.log(math.sqrt(7))

ColoredFractalVisualizer.plot_lsystem_colored(
    gosper_lstring,
    angle_delta=60.0,
    step_length=1.0,
    title=f"Hexagonal Gosper Curve (D≈{gosper_dim:.2f})",
    output_file='output/custom_hexagonal_gosper.png',
    initial_angle=0.0,
    colormap='jet',
    linewidth=1.5
)

print("\nComplete! All colored images have been regenerated.")
print("Color schemes used:")
print("- Koch Snowflake: Purple-yellow gradient (viridis)")
print("- Sierpinski Triangle: Blue-yellow-pink gradient (plasma)")
print("- Hilbert-like: Dark purple-orange gradient (inferno)")
print("- Cantor-Tree: Yellow-green gradient (YlGn)")
print("- Levy C Curve: Cyan-magenta gradient (cool)")
print("- Hexagonal Gosper: Rainbow gradient (jet)")