import math
import numpy as np
from typing import List, Tuple, Dict, Optional
from dataclasses import dataclass

@dataclass
class TopologyTemplate:

    name: str
    n_copies: int
    description: str
    angle_offsets: List[float] 
    
    def __post_init__(self):
        if len(self.angle_offsets) != self.n_copies:
            raise ValueError(
                f"Number of angle offsets ({len(self.angle_offsets)}) "
                f"must equal number of copies ({self.n_copies})"
            )
    
    def is_geometrically_feasible(self, scaling_factor: float) -> bool:
        if scaling_factor > 0.8:
            return False
        
        if len(self.angle_offsets) > 1:
            sorted_angles = sorted(self.angle_offsets)
            min_angle_diff = min(
                sorted_angles[i+1] - sorted_angles[i]
                for i in range(len(sorted_angles) - 1)
            )
            if min_angle_diff < 30 and scaling_factor > 0.6:
                return False
        
        return True
    
    def generate_lsystem_rule(self, predecessor: str = "F") -> str:
        successor = predecessor
        for angle in self.angle_offsets:
            if angle > 0:
                successor += f"[+{angle:.1f}{predecessor}]"
            elif angle < 0:
                successor += f"[{angle:.1f}{predecessor}]"
            else:
                successor += f"[{predecessor}]"
        return successor


class TemplateLibrary:
    
    def __init__(self):
        self.templates: List[TopologyTemplate] = []
        self._initialize_templates()
    
    def _initialize_templates(self):
        self.templates.append(TopologyTemplate(
            name="Linear-2",
            n_copies=2,
            description="Linear Two-Branch (Left-Right)",
            angle_offsets=[-60, 60]
        ))
        
        self.templates.append(TopologyTemplate(
            name="Binary-Symmetric",
            n_copies=2,
            description="Symmetric Binary Tree (45 degrees)",
            angle_offsets=[-45, 45]
        ))
        
        self.templates.append(TopologyTemplate(
            name="Binary-Asymmetric",
            n_copies=2,
            description="Asymmetric Binary Tree",
            angle_offsets=[-30, 60]
        ))
        
        self.templates.append(TopologyTemplate(
            name="Ternary-Star",
            n_copies=3,
            description="Ternary Star (120 degrees distributed)",
            angle_offsets=[-120, 0, 120]
        ))
        
        self.templates.append(TopologyTemplate(
            name="Ternary-Sequential",
            n_copies=3,
            description="Ternary Sequential (60 degree interval)",
            angle_offsets=[-60, 0, 60]
        ))
        
        self.templates.append(TopologyTemplate(
            name="Quaternary-Cross",
            n_copies=4,
            description="Quaternary Cross (90 degrees distributed)",
            angle_offsets=[-90, 0, 90, 180]
        ))
        
        self.templates.append(TopologyTemplate(
            name="Quaternary-Koch",
            n_copies=4,
            description="Koch Curve Type Four-Branch",
            angle_offsets=[-60, -30, 30, 60]
        ))
        
        self.templates.append(TopologyTemplate(
            name="Pentagonal",
            n_copies=5,
            description="Pentagonal Arrangement (72 degrees distributed)",
            angle_offsets=[-144, -72, 0, 72, 144]
        ))
    
    def find_by_n(self, n_copies: int) -> List[TopologyTemplate]:
        return [t for t in self.templates if t.n_copies == n_copies]
    
    def find_by_name(self, name: str) -> Optional[TopologyTemplate]:
        for template in self.templates:
            if template.name == name:
                return template
        return None
    
    def add_template(self, template: TopologyTemplate):
        self.templates.append(template)
    
    def list_all_templates(self) -> List[str]:
        return [t.name for t in self.templates]
    
    def get_template_summary(self) -> str:
        summary = "=== Template Library Summary ===\n"
        summary += f"Total Templates: {len(self.templates)}\n\n"
        
        n_groups = {}
        for template in self.templates:
            n = template.n_copies
            if n not in n_groups:
                n_groups[n] = []
            n_groups[n].append(template)
        
        for n in sorted(n_groups.keys()):
            summary += f"n={n} ({len(n_groups[n])} templates):\n"
            for template in n_groups[n]:
                summary += f"  - {template.name}: {template.description}\n"
            summary += "\n"
        
        return summary


@dataclass
class LSystemRule:
    predecessor: str 
    successor: str 
    probability: float = 1.0 


@dataclass
class GeometricTransform:
    scale: float
    rotation: float
    tx: float = 0.0
    ty: float = 0.0
    
    def apply(self, x: float, y: float) -> Tuple[float, float]:
        x *= self.scale
        y *= self.scale
        cos_r = math.cos(self.rotation)
        sin_r = math.sin(self.rotation)
        x_rot = x * cos_r - y * sin_r
        y_rot = x * sin_r + y * cos_r
        return x_rot + self.tx, y_rot + self.ty


class BoxCountingDimensionCalculator:
    
    @staticmethod
    def calculate_dimension(n_copies: int, scaling_factor: float) -> float:
        if scaling_factor <= 0 or scaling_factor > 1:
            raise ValueError("Scaling factor must be in (0, 1]")
        
        if n_copies <= 0:
            raise ValueError("Number of copies must be positive")
        
        dimension = math.log(n_copies) / math.log(1 / scaling_factor)
        return dimension
    
    @staticmethod
    def calculate_scaling_factor(target_dimension: float, n_copies: int) -> float:
        if target_dimension <= 0:
            raise ValueError("Dimension must be positive")
        
        if n_copies <= 0:
            raise ValueError("Number of copies must be positive")
        
        scaling_factor = math.exp(-math.log(n_copies) / target_dimension)
        return scaling_factor


class FractalLSystemDesigner:
    
    def __init__(self, target_dimension: float):
        self.target_dimension = target_dimension
        self.rules: List[LSystemRule] = []
        self.axiom = ""
        self.transformations: Dict[str, List[GeometricTransform]] = {}
        self.template_library = TemplateLibrary()  # Initialize template library
    
    def design_from_template(self, template: TopologyTemplate) -> Dict:
        scaling_factor = BoxCountingDimensionCalculator.calculate_scaling_factor(
            self.target_dimension, template.n_copies
        )
        
        is_feasible = template.is_geometrically_feasible(scaling_factor)
        
        calculated_dim = BoxCountingDimensionCalculator.calculate_dimension(
            template.n_copies, scaling_factor
        )
        
        print(f"Using Template: {template.name}")
        print(f"  Target Dimension: {self.target_dimension}")
        print(f"  Number of Copies: {template.n_copies}")
        print(f"  Scaling Factor: {scaling_factor:.6f}")
        print(f"  Calculated Dimension: {calculated_dim:.6f}")
        print(f"  Geometric Feasibility: {'✓ Feasible' if is_feasible else '✗ Possible Overlap'}")
        
        self.axiom = "F"
        successor = template.generate_lsystem_rule("F")
        self.rules = [LSystemRule("F", successor)]
        
        self.transformations["F"] = []
        for angle_offset in template.angle_offsets:
            transform = GeometricTransform(
                scale=scaling_factor,
                rotation=math.radians(angle_offset)
            )
            self.transformations["F"].append(transform)
        
        return {
            "axiom": self.axiom,
            "rules": self.rules,
            "scaling_factor": scaling_factor,
            "target_dimension": self.target_dimension,
            "calculated_dimension": calculated_dim,
            "n_copies": template.n_copies,
            "template_name": template.name,
            "geometrically_feasible": is_feasible,
            "transformations": self.transformations
        }
    
    def find_all_designs(self, n_range: Tuple[int, int] = (2, 5)) -> List[Dict]:
        designs = []
        
        for n in range(n_range[0], n_range[1] + 1):
            templates = self.template_library.find_by_n(n)
            
            for template in templates:
                design = self.design_from_template(template)
                designs.append(design)
                print("-" * 60)
        
        return designs
    
    def design_koch_variant(self, n_copies: int, rotation_angle: float = 60) -> Dict:
        scaling_factor = BoxCountingDimensionCalculator.calculate_scaling_factor(
            self.target_dimension, n_copies
        )
        
        calculated_dim = BoxCountingDimensionCalculator.calculate_dimension(
            n_copies, scaling_factor
        )
        
        print(f"Design Parameters:")
        print(f"  Target Dimension: {self.target_dimension}")
        print(f"  Number of Copies: {n_copies}")
        print(f"  Scaling Factor: {scaling_factor:.6f}")
        print(f"  Calculated Dimension: {calculated_dim:.6f}")
        print(f"  Error: {abs(calculated_dim - self.target_dimension):.2e}")
        
        self.axiom = "F"
        rotation_rad = math.radians(rotation_angle)
        
        if n_copies == 2:
            self.rules = [LSystemRule("F", f"F[+{rotation_angle}F][-{rotation_angle}F]")]
        elif n_copies == 3:
            angle_step = 360 / n_copies
            successor = "F"
            for i in range(n_copies):
                angle = (i - n_copies // 2) * angle_step
                successor += f"[+{angle}F]"
            self.rules = [LSystemRule("F", successor)]
        elif n_copies == 4:
            angle = 90
            self.rules = [
                LSystemRule("F", f"F[+{angle}F][-{angle}F][+{angle}F][+{angle}F]")
            ]
        else:
            angle_step = 360 / n_copies
            successor = "F"
            for i in range(n_copies):
                angle = (i - n_copies // 2) * angle_step
                successor += f"[+{angle}F]"
            self.rules = [LSystemRule("F", successor)]

        self.transformations["F"] = []
        for i in range(n_copies):
            angle_offset = (i - n_copies // 2) * math.radians(rotation_angle)
            transform = GeometricTransform(
                scale=scaling_factor,
                rotation=angle_offset
            )
            self.transformations["F"].append(transform)
        
        return {
            "axiom": self.axiom,
            "rules": self.rules,
            "scaling_factor": scaling_factor,
            "target_dimension": self.target_dimension,
            "calculated_dimension": calculated_dim,
            "n_copies": n_copies,
            "transformations": self.transformations
        }
    
    def design_cantor_variant(self, n_copies: int) -> Dict:
        scaling_factor = BoxCountingDimensionCalculator.calculate_scaling_factor(
            self.target_dimension, n_copies
        )
        
        calculated_dim = BoxCountingDimensionCalculator.calculate_dimension(
            n_copies, scaling_factor
        )
        
        print(f"Design Parameters (Cantor Type):")
        print(f"  Target Dimension: {self.target_dimension}")
        print(f"  Number of Copies: {n_copies}")
        print(f"  Scaling Factor: {scaling_factor:.6f}")
        print(f"  Calculated Dimension: {calculated_dim:.6f}")
        
        self.axiom = "A"
        
        if n_copies == 2:
            self.rules = [LSystemRule("A", "A-A")]
        else:
            successor = "A"
            for i in range(n_copies - 1):
                successor += "-A"
            self.rules = [LSystemRule("A", successor)]
        
        return {
            "axiom": self.axiom,
            "rules": self.rules,
            "scaling_factor": scaling_factor,
            "target_dimension": self.target_dimension,
            "calculated_dimension": calculated_dim,
            "n_copies": n_copies
        }
    
    def get_dimension_range(self, n_copies: int) -> Tuple[float, float]:
        min_dim = math.log(n_copies) / math.log(1 / 0.01)  # Approximation
        
        max_dim = math.log(n_copies)  # As r -> 0, ln(1/r) -> inf, but practically dimension grows
        
        return min_dim, max_dim


class LSystemInterpreter:
    
    def __init__(self, angle_delta: float = 90.0):
        self.angle_delta = math.radians(angle_delta)
    
    def interpret(self, lstring: str, initial_position: Tuple[float, float] = (0, 0),
                  initial_angle: float = 90.0) -> List[Tuple[float, float]]:
        x, y = initial_position
        angle = math.radians(initial_angle)
        stack = []
        coordinates = [(x, y)]
        
        for char in lstring:
            if char == 'F' or char == 'A':
                x += math.cos(angle)
                y += math.sin(angle)
                coordinates.append((x, y))
            elif char == '+':
                angle += self.angle_delta
            elif char == '-':
                angle -= self.angle_delta
            elif char == '[':
                stack.append((x, y, angle))
            elif char == ']':
                if stack:
                    x, y, angle = stack.pop()
                    coordinates.append((x, y))
        
        return coordinates
    
    @staticmethod
    def generate_lstring(axiom: str, rules: List[LSystemRule], iterations: int) -> str:
        current = axiom
        rule_dict = {rule.predecessor: rule.successor for rule in rules}
        
        for _ in range(iterations):
            next_string = ""
            for char in current:
                if char in rule_dict:
                    next_string += rule_dict[char]
                else:
                    next_string += char
            current = next_string
        
        return current


def main():
    
    print("=" * 70)
    print("Design Fractal L-Systems based on Box-Counting Dimension (Using Template Library)")
    print("=" * 70)
    
    print("\n[Example 0] Template Library Overview")
    print("-" * 70)
    library = TemplateLibrary()
    print(library.get_template_summary())
    
    print("\n[Example 1] Design fractal with dimension approx 1.5 using templates")
    print("-" * 70)
    
    target_dim_1 = 1.5
    designer1 = FractalLSystemDesigner(target_dim_1)
    
    for template_name in ["Linear-2", "Binary-Symmetric", "Binary-Asymmetric"]:
        template = library.find_by_name(template_name)
        if template:
            result = designer1.design_from_template(template)
            print(f"Generated L-System Rule: {result['rules'][0].predecessor} -> {result['rules'][0].successor}")
            print("-" * 60)
    
    print("\n[Example 2] Iterate all templates to find feasible designs for dimension 1.8")
    print("-" * 70)
    
    target_dim_2 = 1.8
    designer2 = FractalLSystemDesigner(target_dim_2)
    all_designs = designer2.find_all_designs(n_range=(2, 4))
    
    feasible_designs = [d for d in all_designs if d['geometrically_feasible']]
    print(f"\nFound {len(feasible_designs)} geometrically feasible designs")
    
    print("\n" + "=" * 70)
    print("[Example 3] Verify standard Koch curve dimension (should be approx 1.26)")
    print("-" * 70)
    
    koch_dim = BoxCountingDimensionCalculator.calculate_dimension(n_copies=4, scaling_factor=1/3)
    print(f"Standard Koch Curve Dimension: {koch_dim:.6f}")
    print(f"Theoretical Value (ln(4)/ln(3)) = {math.log(4)/math.log(3):.6f}")
    
    print("\n" + "=" * 70)
    print("[Example 4] Sierpinski Triangle (Dimension approx 1.585)")
    print("-" * 70)
    
    sierpinski_dim = BoxCountingDimensionCalculator.calculate_dimension(n_copies=3, scaling_factor=0.5)
    print(f"Sierpinski Triangle Dimension: {sierpinski_dim:.6f}")
    print(f"Theoretical Value (ln(3)/ln(2)) = {math.log(3)/math.log(2):.6f}")
    
    print("\n" + "=" * 70)
    print("[Example 5] Create and use custom template")
    print("-" * 70)
    
    custom_template = TopologyTemplate(
        name="Custom-Hexagonal",
        n_copies=6,
        description="Hexagonal Arrangement (60 degrees distributed)",
        angle_offsets=[-150, -90, -30, 30, 90, 150]
    )
    
    library.add_template(custom_template)
    print(f"Added custom template: {custom_template.name}")
    print(f"Angle offsets: {custom_template.angle_offsets}")
    
    designer5 = FractalLSystemDesigner(1.6)
    result5 = designer5.design_from_template(custom_template)
    print(f"Generated L-System Rule: {result5['rules'][0].predecessor} -> {result5['rules'][0].successor}")
    
    print("\n" + "=" * 70)
    print("[Example 6] Generate L-System string and coordinates")
    print("-" * 70)
    
    designer6 = FractalLSystemDesigner(1.5)
    template6 = library.find_by_name("Binary-Symmetric")
    result6 = designer6.design_from_template(template6)
    
    interpreter = LSystemInterpreter(angle_delta=90.0)
    
    for iteration in range(1, 4):
        lstring = interpreter.generate_lstring(
            axiom=result6['axiom'],
            rules=result6['rules'],
            iterations=iteration
        )
        print(f"\nIteration {iteration}:")
        print(f"  String Length: {len(lstring)}")
        print(f"  String: {lstring[:80]}{'...' if len(lstring) > 80 else ''}")
        
        coords = interpreter.interpret(lstring, initial_position=(0, 0), initial_angle=90)
        print(f"  Generated Coordinate Points: {len(coords)}")


if __name__ == "__main__":
    main()